/**
*
* \file        cnet_pnp.c
*
* \brief       Implements "Plug-N-Play/TSID" according to "Plug And Play - Cresnet
*              Commands & Oper.doc" - rev: 10/12/05
*
* \author      Larry Salant (adapted from Cajita - Pete McCormick (copied from others))
*
* \date        12/18/2008
*
*/

#include "Cresnet.h"
#include <string.h>                     // for memmove
#include "errors.h"
#include "hardware.h"                   // indirectly used by TimeNet
#include <stdlib.h>                     // for memmove
#include "console.h"
#include "uart_cresnet_slave.hpp"
#include "cresnet_slave.h"
#include "cnet_pnp.h"
#include "os.h"
#include "leds.h"

////////////////////////////////////////////////////////////////////////////////

// CNET_PLUG_PLAY_PNP_SET_STATE_CMD sub-commands
#define	CNET_PLUG_PLAY_SET_RESPONSDTOMASK		0x01
#define	CNET_PLUG_PLAY_CLR_RESPONDTOMASK		0x02
#define	CNET_PLUG_PLAY_SET_LIGHTANDPOLL			0x03
#define	CNET_PLUG_PLAY_CLR_LIGHTANDPOLL			0x04
#define	CNET_PLUG_PLAY_CLR_LIGHTANDPOLL_WCFM	0x05
#define	CNET_PLUG_PLAY_CLR_REQSELFINSTALL		0x0A

// CNET_PLUG_PLAY_RESPOND_NOW_CMD sub-commands
#define CNET_PLUG_PLAY_RESPOND_WITH_BREAK		0x01
#define CNET_PLUG_PLAY_RESPOND_WITH_IDENTITY	0x02

#define	CNET_PLUG_PLAY_BROADCAST_PPN			0x5A5A5A5A

#define PNP_SETUP_TIMER_MS 250
#define PNP_SETUP_SHOW_TIMER_MS 1000
#define PNP_SELF_INSTALL_TIMER_MS 60000

static void *TouchSettableBlinkTimer;
static void *TouchSettableBlinkShowTimer;
#ifdef SELF_INSTALL_SUPPORTED
static void *RequestSelfInstallTimer;
#endif

////////////////////////////////////////////////////////////////////////////////

typedef struct
{
  BOOL ValidCresnet;
  BOOL RespondToMask;
  BOOL LightAndPoll;
  BOOL ButtonPressed;
  BOOL Confirm;
  BOOL RequestSelfInstall;
} PNPDEVICESTATES;

////////////////////////////////////////////////////////////////////////////////

void PNPRxParse(UINT8* packet, UINT8 type);
void ClearLightAndPoll(void);

////////////////////////////////////////////////////////////////////////////////

static PNPDEVICESTATES PNPStates;
////////////////////////////////////////////////////////////////////////////////

void HideRequestSelfInstallDisplay(UINT8 ignore)
{
#ifdef SELF_INSTALL_SUPPORTED
  /* disable the Request Self Install duration timer */
  NU_Control_Timer(&RequestSelfInstallTimer, NU_DISABLE_TIMER);

  /* if we are currently in the request Self Install state */
  if (PNPStates.RequestSelfInstall)
  {
    /* Hide the Request Self Install Display by going to the previous page */
    //change_active_screen(PREVIOUS_PAGE_POINTER);
    PNPStates.RequestSelfInstall = 0;
  }

  /* restore the cresnet ID */
  CresnetParamsRestore();
#endif
  if(GetCresnetSlaveUart()->UartCresnetSlaveGetID() != CNET_ID_PPN_ADDRESSING)
  {
    PNPStates.ValidCresnet = 1;
  }
}

void TSIDBlink(UINT8 ignore)
{
  static UINT8 led_state;

  led_state = !led_state;
  //dig_set_in(TOUCH_SETTABLE_ID_BUTTON, button_state);
  PPNSetupLED(led_state);
  // Sample the button only if in !button_pressed state
  if(PNPStates.LightAndPoll == 1)
  {
    if( PNPStates.ButtonPressed == 1 )
    {
        PPNSetupLED(0); // Turn off LED.
    }
  }
}

static void TSIDBlinkShow(UINT8 ignore)
{
  // Sanity check
  if( PNPStates.Confirm ==  1)
  {
    OsCancelTimer(TouchSettableBlinkShowTimer);
    PPNSetupLED(0);
    PNPStates.Confirm = 0;
  }
}


void PNPInit(void)
{
   pfClearLightAndPoll = ClearLightAndPoll;
   pfCheckTouchSettableIDPacket = CheckTouchSettableIDPacket;
   pfPNPRxParse = PNPRxParse;
#ifdef SELF_INSTALL_SUPPORTED
  STATUS status;

  if (NU_Create_Timer(&RequestSelfInstallTimer,"INST_Tmr",
    HideRequestSelfInstallDisplay,0,6000,0,NU_DISABLE_TIMER) != NU_SUCCESS)
  {
    DmSystemError(FATAL_APPINIT);
  }
#endif
  PPNSetupLED(0);                     // Turn off LED.

  PNPStates.ValidCresnet = (GetCresnetSlaveUart()->UartCresnetSlaveGetID() != CNET_ID_PPN_ADDRESSING);
}

static UINT32 GeneratePPNNumber(void)
{
  UINT32 new_PPN_number;

  UINT8 high_byte;
  UINT8 second_byte;
  UINT8 third_byte;
  UINT8 low_byte;

  high_byte = 0xF0;
  second_byte = rand() & 0x07F;
  third_byte = rand() & 0x0FF;
  low_byte = rand() & 0x0FF;

  /* assemble the PPN number */
  new_PPN_number = (high_byte << 24) | (second_byte << 16) | (third_byte << 8) | low_byte;

  return new_PPN_number;
}

static void LightAndPollStateIndication(int show_it)
{
  if (show_it)
  {
    /* start the timer */
    TouchSettableBlinkTimer = OsStartTimer(PNP_SETUP_TIMER_MS, (void*)TSIDBlink, OS_TIMER_PERIODIC);
    InitSetUpLED_Driver();
  }
  else
  {
    /* kill the timer */
    OsCancelTimer(TouchSettableBlinkTimer);
    PPNSetupLED(0); // Turn off LED.
    GiveupSetUpLED_Driver();
  }
}

void TSIDClearPollFunction(void)
{
  if(PNPStates.LightAndPoll == 1)
  {
    if(PNPStates.ButtonPressed == 0)
    {
      PNPStates.ButtonPressed= 1;
    }
  }
}

BOOL PNPCheckstate(UINT8 c)
{
  struct
  {
    BOOL* variable;
    BOOL  state;
  } static ppn_validation_table[] = {
    {NULL,1},                            //  0
    {&PNPStates.RespondToMask,1},       //  1
    {&PNPStates.RespondToMask,0},      //  2
    {&PNPStates.LightAndPoll, 1},       //  3
    {&PNPStates.LightAndPoll, 0},      //  4
    {&PNPStates.ButtonPressed,1},       //  5
    {&PNPStates.ButtonPressed,0},      //  6
    {NULL,0},                           //  7
    {NULL,0},                           //  8
    {&PNPStates.RequestSelfInstall,1},  //  9
    {&PNPStates.RequestSelfInstall,0}, // 10
    {&PNPStates.ValidCresnet,      1},  // 11
    {&PNPStates.ValidCresnet,      0}, // 12
  };
  if (c == 0)
    return 1;
  else if (c > 12)
    return 0;
  else if (c == 5)
    return PNPStates.ButtonPressed && PNPCheckstate(3);   // button pressed can only be true if in light&poll state
  return ppn_validation_table[c].state ? *ppn_validation_table[c].variable : !*ppn_validation_table[c].variable;
}

/*
returns:
0 - if the packet is not approved
1 - if the packet is approved
*/
// ValidatePPNAddr
BOOL IsPPNValid(UINT32 PPN_number,UINT8 PPN_address_qualifier)
{
  BOOL          qualifier_approval;
  BOOL          PPN_number_approval = 0;     // default = failed
  UINT8         qualifier           = PPN_address_qualifier & 0x7F;

  if (qualifier == 7)
    qualifier_approval = PNPCheckstate(3) && PNPCheckstate(6);
  else
    qualifier_approval = PNPCheckstate(qualifier);

  if (PPN_number == UartCresnetSlaveGetPPNAddr())
    PPN_number_approval = 1;
  else if (PPN_number == CNET_PLUG_PLAY_BROADCAST_PPN)
  {
    if (PPN_address_qualifier & 0x80)
      PPN_number_approval = 1;
  }
  return qualifier_approval && PPN_number_approval;
}

int TSIDTestCmd(int ignore, char *cmd)
{
  TSIDClearPollFunction();
  return 0;
}
/**
 * \author      Larry Salant
 * \date        12/23/2008
 * \brief       generate break if our PPN address match the mask sent in
 * \param       mask sent from control system to compare against
 * \return      void
 * \retval      none
 **/
void ProcessPPNMask(UINT32 PPN_mask)
{
  // special case

  if (PPN_mask == 0 && UartCresnetSlaveGetPPNAddr() == 0)
  {
    GetCresnetSlaveUart()->UartCresnetSlaveGenerateBreak();
  }
  else
  {
    // find the lowest order bit in the mask
    //  Algorithm: starting from LSBit in mask the first 1 encountered
    //  will mark the beginning of range comparison, e.g. if adr_bin =  0101 1001, mask = 0001 0000
    //  1. find the rightmost 1 in the mask : position 4 (zero based)
    // 2. starting from the MSBit all the way to rightmost 1 in the mask adr bits and msk bits must match
    UINT32 tmp_mask = 0xffffffff;
    UINT32 mask_bit = 1;

    while ( tmp_mask )
    {
        if ( mask_bit & PPN_mask )
            break;              // find the rightmost 1 in the mask

        tmp_mask <<= 1;
        mask_bit <<= 1;
    }

    if (( tmp_mask ) && (( UartCresnetSlaveGetPPNAddr() & tmp_mask ) == PPN_mask ))
    {
      GetCresnetSlaveUart()->UartCresnetSlaveGenerateBreak();
    }
  }
}

/*
return codes:

-1 - packet rejected.  Not for this device
1 - packet handled, no further processing necessary
2 - packet accepted, re-process type 0x19 payload
*/
INT8 CheckTouchSettableIDPacket(UINT8* packet)
{
  UINT32 PPN_number = BE2NATIVE32(*(UINT32*) &packet[3]);

  if (packet[2] == PPN_ENCAP_PKT)
  {
    if (IsPPNValid(PPN_number,packet[7]))
      return 2;
  }
  else if (packet[2] == PNP_MASK)
  {
    if (PNPStates.RespondToMask && packet[1] == 5)
    {
      ProcessPPNMask(PPN_number);
      return 1;
    }
  }
  return -1;
}

void ClearLightAndPoll(void)
{
  HideRequestSelfInstallDisplay(0);
  PNPStates.ButtonPressed = 0;
  if(PNPStates.LightAndPoll)
  {
    PNPStates.LightAndPoll = 0;
    LightAndPollStateIndication(0);
  }
}

/**
 * \author      Roni Hudes
 * \brief       Return if device is in TSID mode.
 * \date        09/11/2009
 * \return      BOOL status
 * \param       void
 */
BOOL LightAndPollStatus (void)
{
  return (PNPStates.LightAndPoll);
}

void ShowConfirmationDisplay(void)
{
  PPNSetupLED(1);
  TouchSettableBlinkShowTimer = OsStartTimer(PNP_SETUP_SHOW_TIMER_MS, (void*)TSIDBlinkShow, OS_TIMER_PERIODIC);
}
/**
 * \author      Larry Salant
 * \brief       called when the user has pressed the setup button
 * \date        12/23/2008
 * \return      void
 * \retval      void
 * \param       void
 */
void PPNSetupButtonPressed(void)
{
  // button was pressed.
  PNPStates.ButtonPressed = 1;
}

void CresnetParsePNPSetStateCmd(UINT8 * packet)
{
  switch (packet[3])
  {
  case CNET_PLUG_PLAY_SET_RESPONSDTOMASK:            // Set RespondToMask
    PNPStates.RespondToMask = 1;
    break;

  case CNET_PLUG_PLAY_CLR_RESPONDTOMASK:            // Clear RespondToMask
    PNPStates.RespondToMask = 0;
    break;

  case CNET_PLUG_PLAY_SET_LIGHTANDPOLL:            // Set LightAndPoll
    HideRequestSelfInstallDisplay(0);
    if(PNPStates.LightAndPoll == 0)
    {
      PNPStates.LightAndPoll = 1;
      PNPStates.ButtonPressed = 0;                 // clear any previous presses
      LightAndPollStateIndication(1);
    }
    break;

  case CNET_PLUG_PLAY_CLR_LIGHTANDPOLL:            // Clear LightAndPoll
    ClearLightAndPoll();
    break;

  case CNET_PLUG_PLAY_CLR_LIGHTANDPOLL_WCFM:            // Clear LightAndPoll w/ Confirm
    ClearLightAndPoll();
    PNPStates.Confirm = 1;
    ShowConfirmationDisplay();
    break;

  case CNET_PLUG_PLAY_CLR_REQSELFINSTALL:            // Clear RequestSelfInstall
    HideRequestSelfInstallDisplay(0);
    break;
  }
}

void CresnetParsePPNSetPPNNoCmd(UINT8 * packet)
{
  SetPNPNumber(BE2NATIVE32(*(UINT32*)&packet[3]));
}

void CresnetParsePPNSetSelfPPNCmd(UINT8 * packet)
{
  UINT32 value;

  value = GeneratePPNNumber();
  SetPNPNumber(value);
}

void CresnetParseTempNetId(UINT8 * packet)
{
  // packet format = <id> <02> <fb> <new_id>
  /* Temporarily sets device's cresnet ID to a new address.
  * It only does the ram copy and not the
  * EEPROM copy so that the unit would always come up
  * at the same address.  See notes concerning NEW_NET_ID below.
  */
  if (packet[3] == 1 || (packet[3] > 2 && packet[3] < 0xFF))
  {
    GetCresnetSlaveUart()->UartCresnetSlaveSetTempID(*(packet+3));
  }
}

void CresnetParsePNPChangeIdCmd(UINT8 * packet)
{
  UINT32 id;
  CDMUartCresnetSlave *pSlave = GetCresnetSlaveUart(); // get pointer to the current

  id = packet[3];

  if (id == 1 || (id > 2 && id < 255))
  {
    pSlave->UartCresnetSlaveSetID(*(packet+3));
    PNPStates.ValidCresnet = ((pSlave->UartCresnetSlaveGetID() > 2) && (pSlave->UartCresnetSlaveGetID() < 255));
    pSlave->UartCresnetSlaveSetRestartFlag(0);
  }
  else
  {
    DmConsolePrintf("Commanded to change ID to invalid value: %2X",packet[3]);
  }
}

/**
 * \author      Larry Salant
 * \date        12/18/2008
 * \brief       Parses Plug and Play cresnet packets (if card supports it)
 * \param       pointer to packet
 * \param       packet type
 * \return      void
 * \retval      none
 */
void PNPRxParse(UINT8* packet, UINT8 type)
{
    switch(type)
    {
        case PNP_SET_STATE:
            CresnetParsePNPSetStateCmd(packet);
            break;

        case PNP_SET_BY_CNET_ID:
            CresnetParsePPNSetPPNNoCmd(packet);
            break;

        case PNP_SET_BY_RANDOMIZE:
            CresnetParsePPNSetSelfPPNCmd(packet);
            break;

        case TEMP_NET_ID:
            CresnetParseTempNetId(packet);
            break;

        case NEW_NET_ID:
            CresnetParsePNPChangeIdCmd(packet);
            break;

        default:
            break;
    }
}
